const connectionManager = require('../lib/manager/websocket-manager');
const { NodeHelpers, NodePatterns } = require('../lib/utils/node-helpers');

module.exports = function (RED) {
    function iobInInject(config) {
        RED.nodes.createNode(this, config);
        const node = this;
        const { setStatus, setError } = NodeHelpers.createStatusHelpers(node);

        const serverConfig = NodeHelpers.validateServerConfig(RED, config, setError);
        if (!serverConfig) return;

        const { globalConfig, connectionDetails, serverId } = serverConfig;

        // Helper to resolve env vars
        function resolveEnvVar(value) {
            if (typeof value === 'string' && value.includes('${')) {
                return value.replace(/\${([^}]+)}/g, (match, varName) => {
                    const envValue = RED.util.evaluateNodeProperty(varName, 'env', node);
                    return envValue !== undefined ? envValue : match;
                });
            }
            return value;
        }

        // Initialize state
        let subscriptionPattern = config.state ? resolveEnvVar(config.state.trim()) : '';
        
        // Determine initial mode
        let inputMode = 'single';
        
        let isWildcardPattern = subscriptionPattern.includes('*');
        let isMultipleStates = false;
        let isSingleState = !isWildcardPattern;

        const settings = {
            outputProperty: config.outputProperty?.trim() || "payload",
            ackFilter: config.ackFilter || "both",
            sendInitialValue: config.sendInitialValue && !isWildcardPattern,
            outputMode: "individual", // Fixed for now, could be dynamic later
            filterMode: config.filterMode || "all",
            enableExternalTrigger: false,
            triggerGroup: config.triggerGroup || "iobroker_in_nodes",
            serverId,
            nodeId: node.id,
            useWildcard: isWildcardPattern,
            inputMode: inputMode
        };

        node.currentConfig = connectionDetails;
        node.isInitialized = false;
        node.isSubscribed = false;
        
        // Dynamic properties
        node.subscriptionPattern = subscriptionPattern;
        node.stateList = [];
        node.inputMode = inputMode; // 'single' or 'multiple'
        
        node.currentStateValues = new Map();
        node.subscribedStates = new Set();
        node.lastValue = undefined;
        node.hasReceivedValue = false;
        node.previous = new Map();

        function updateStatusWithValue(isInitialValue = false) {
            if (node.inputMode === 'single' && isSingleState && node.hasReceivedValue && node.lastValue !== undefined) {
                const formattedValue = NodeHelpers.formatValueForStatus(node.lastValue);
                const filterLabel = (settings.filterMode === 'changes-only' || settings.filterMode === 'changes-smart') ? ' [Changes]' : '';
                const initialLabel = isInitialValue ? ' (initial)' : '';
                setStatus("green", "dot", `${formattedValue}${filterLabel}${initialLabel}`);
            } else {
                const timestamp = new Date().toLocaleTimeString(undefined, { hour12: false });
                let statusText;

                if (node.inputMode === 'multiple') {
                     const subscribedCount = node.subscribedStates.size;
                     const filterLabel = (settings.filterMode === 'changes-only' || settings.filterMode === 'changes-smart') ? ' [Changes]' : '';
                     if (node.hasReceivedValue) {
                         statusText = `${subscribedCount} states${filterLabel} - Last: ${timestamp}`;
                     } else {
                         statusText = `${node.stateList.length} states${filterLabel}`;
                     }
                } else if (isWildcardPattern) {
                    const filterLabel = (settings.filterMode === 'changes-only' || settings.filterMode === 'changes-smart') ? ' [Changes]' : '';
                    if (node.hasReceivedValue) {
                        statusText = `Pattern active${filterLabel} - Last: ${timestamp}`;
                    } else {
                        statusText = `Pattern: ${node.subscriptionPattern}${filterLabel}`;
                    }
                } else {
                    const filterLabel = (settings.filterMode === 'changes-only' || settings.filterMode === 'changes-smart') ? ' [Changes]' : '';
                    if (node.subscriptionPattern) {
                         statusText = `Ready: ${node.subscriptionPattern}${filterLabel}`;
                    } else {
                         statusText = `Waiting for input...`;
                         setStatus("blue", "dot", statusText);
                         return;
                    }
                }
                setStatus("green", "dot", statusText);
            }
        }

        // Use NodeHelpers for standard validation and error handling
        function shouldSendMessage(ack, filter) {
            return NodeHelpers.shouldSendByAck(ack, filter);
        }

        function shouldSendByValue(stateId, newValue, filterMode, isInitialValue = false) {
            if (isInitialValue) {
                return true;
            }

            if (filterMode !== 'changes-only' && filterMode !== 'changes-smart') {
                return true;
            }

            const previousValue = node.previous.get(stateId);

            if (filterMode === 'changes-only' && previousValue === undefined) {
                return true;
            }

            if (typeof newValue === 'object' && newValue !== null) {
                const currentJSON = JSON.stringify(newValue);
                const previousJSON = previousValue !== undefined ? JSON.stringify(previousValue) : undefined;
                return currentJSON !== previousJSON;
            }

            return newValue !== previousValue;
        }

        function updatePreviousValue(stateId, value) {
            if (settings.filterMode === 'changes-only' || settings.filterMode === 'changes-smart') {
                if (typeof value === 'object' && value !== null) {
                    try {
                        node.previous.set(stateId, JSON.parse(JSON.stringify(value)));
                    } catch (e) {
                        node.previous.set(stateId, value);
                    }
                } else {
                    node.previous.set(stateId, value);
                }
            }
        }

        async function initializeSmartChangeFilter() {
            if (settings.filterMode !== 'changes-smart') {
                return;
            }

            try {
                if (node.inputMode === 'multiple') {
                    for (const stateId of node.stateList) {
                        try {
                            const state = await connectionManager.getState(settings.serverId, stateId);
                            if (state && state.val !== undefined) {
                                updatePreviousValue(stateId, state.val);
                                node.debug(`Smart filter: Pre-loaded ${stateId} = ${state.val}`);
                            }
                        } catch (error) {
                            node.debug(`Smart filter: Could not load ${stateId}: ${error.message}`);
                        }
                    }
                } else if (!isWildcardPattern && node.subscriptionPattern) {
                    try {
                        const state = await connectionManager.getState(settings.serverId, node.subscriptionPattern);
                        if (state && state.val !== undefined) {
                            updatePreviousValue(node.subscriptionPattern, state.val);
                            node.debug(`Smart filter: Pre-loaded ${node.subscriptionPattern} = ${state.val}`);
                        }
                    } catch (error) {
                        node.debug(`Smart filter: Could not load ${node.subscriptionPattern}: ${error.message}`);
                    }
                }
            } catch (error) {
                node.warn(`Smart change filter initialization failed: ${error.message}`);
            }
        }

        function createMessage(stateId, state, isInitialValue = false) {
            const message = {
                topic: stateId,
                state: state,
                timestamp: Date.now()
            };

            if (node.inputMode === 'single' && isWildcardPattern) {
                message.pattern = node.subscriptionPattern;
            }

            if (isInitialValue) {
                message.initial = true;
            }

            RED.util.setMessageProperty(message, settings.outputProperty, state.val);
            return message;
        }

        function onStateChange(stateId, state, isInitialValue = false) {
            try {
                if (!state || state.val === undefined) {
                    node.warn(`Invalid state data received for ${stateId}`);
                    return;
                }

                if (!shouldSendMessage(state.ack, settings.ackFilter)) {
                    return;
                }

                if (node.inputMode === 'multiple' && !node.subscribedStates.has(stateId)) {
                    return;
                }

                if (!shouldSendByValue(stateId, state.val, settings.filterMode, isInitialValue)) {
                    node.debug(`Change filter blocked duplicate value for ${stateId}: ${state.val}`);
                    return;
                }

                if (!(isInitialValue && settings.filterMode === 'changes-smart')) {
                    updatePreviousValue(stateId, state.val);
                }

                if (node.inputMode === 'single' && isSingleState) {
                    node.lastValue = state.val;
                    node.hasReceivedValue = true;
                    // Also store in currentStateValues for caching
                    node.currentStateValues.set(stateId, state);
                } else if (node.inputMode === 'multiple') {
                    node.currentStateValues.set(stateId, state);
                }

                // Helper function to send message and update status
                function sendMessageAndUpdateStatus(message, isInitial = false) {
                    node.send(message);
                    node.hasReceivedValue = true;
                    updateStatusWithValue(isInitial);
                }

                sendMessageAndUpdateStatus(createMessage(stateId, state, isInitialValue), isInitialValue);

            } catch (error) {
                node.error(`State change processing error: ${error.message}`);
                setError(`Processing error: ${error.message}`, "Process error");
            }
        }

        function createCallback(forceInitialValue = false) {
            const callback = onStateChange;

            callback.wantsInitialValue = settings.sendInitialValue || forceInitialValue;

            callback.onInitialValue = function (stateId, state) {
                try {
                    if (!shouldSendMessage(state.ack, settings.ackFilter)) {
                        return;
                    }

                    if (settings.filterMode !== 'changes-smart') {
                        updatePreviousValue(stateId, state.val);
                    }

                    if (node.inputMode === 'single' && isSingleState) {
                        node.lastValue = state.val;
                        node.hasReceivedValue = true;
                        node.currentStateValues.set(stateId, state);
                    } else if (node.inputMode === 'multiple') {
                        node.currentStateValues.set(stateId, state);
                    }

                    const message = createMessage(stateId, state, true);
                    node.send(message);
                    updateStatusWithValue(true);

                } catch (error) {
                    node.error(`Initial value processing error: ${error.message}`);
                }
            };

            const statusTexts = {
                ready: isWildcardPattern
                        ? `Pattern: ${node.subscriptionPattern}${(settings.filterMode === 'changes-only' || settings.filterMode === 'changes-smart') ? ' [Changes]' : ''}`
                        : `Ready${(settings.filterMode === 'changes-only' || settings.filterMode === 'changes-smart') ? ' [Changes]' : ''}`,
                disconnected: "Disconnected"
            };

            const baseCallback = NodeHelpers.createSubscriptionEventCallback(
                node,
                setStatus,
                () => {
                    node.isSubscribed = true;
                    // Sync subscribedStates for multiple mode
                    if (node.inputMode === 'multiple') {
                         node.subscribedStates.clear();
                         node.stateList.forEach(s => node.subscribedStates.add(s));
                    }
                },
                statusTexts
            );

            Object.assign(callback, baseCallback);
            return callback;
        }

        async function subscribeToStates(forceInitialValue = false) {
            if (node.inputMode === 'single' && !node.subscriptionPattern) {
                setStatus("blue", "dot", "Waiting for input...");
                return;
            }
            if (node.inputMode === 'multiple' && node.stateList.length === 0) {
                 setStatus("blue", "dot", "Waiting for input (empty list)...");
                 return;
            }

            const callback = createCallback(forceInitialValue);

            if (node.inputMode === 'multiple') {
                try {
                    const successfulStates = await connectionManager.subscribeMultiple(
                        settings.nodeId,
                        settings.serverId,
                        node.stateList,
                        callback,
                        globalConfig
                    );
                    
                    node.subscribedStates.clear();
                    successfulStates.forEach(s => node.subscribedStates.add(s));
                    
                } catch (error) {
                    const isAuthError = error.message && (error.message.includes('Authentication failed') || error.message.includes('auth_failed'));
                    if (!isAuthError) {
                        node.error(`Failed to subscribe to multiple states: ${error.message}`);
                    }
                    throw error;
                }
            } else {
                await connectionManager.subscribe(
                    settings.nodeId,
                    settings.serverId,
                    node.subscriptionPattern,
                    callback,
                    globalConfig
                );
            }
        }

        async function triggerCurrentValues() {
            node.debug("Triggering manual initial value send");
            
            if (node.inputMode === 'multiple') {
                for (const stateId of node.stateList) {
                    try {
                        const state = await connectionManager.getState(settings.serverId, stateId);
                        if (state && state.val !== undefined) {
                             const message = createMessage(stateId, state, true);
                             node.send(message);
                             updateStatusWithValue(true);
                        }
                    } catch (error) {
                        node.warn(`Could not trigger value for ${stateId}: ${error.message}`);
                    }
                }
            } else if (node.inputMode === 'single' && !isWildcardPattern && node.subscriptionPattern) {
                try {
                    const state = await connectionManager.getState(settings.serverId, node.subscriptionPattern);
                    if (state && state.val !== undefined) {
                         const message = createMessage(node.subscriptionPattern, state, true);
                         node.send(message);
                         updateStatusWithValue(true);
                    }
                } catch (e) {
                     node.warn(`Could not trigger value for ${node.subscriptionPattern}: ${e.message}`);
                }
            } else {
                // Wildcard or empty
                if (isWildcardPattern) {
                    node.warn("Triggering initial value not supported for wildcard patterns");
                }
            }
        }

        async function initialize() {
            const status = connectionManager.getConnectionStatus(settings.serverId);
            if (node.isSubscribed && status.connected && status.ready) {
                return;
            }

            try {
                setStatus("yellow", "ring", "Connecting...");

                node.currentStateValues.clear();
                node.subscribedStates.clear();
                node.hasReceivedValue = false;
                node.lastValue = undefined;
                node.previous.clear();

                await NodeHelpers.handleConfigChange(node, config, RED, settings);

                await subscribeToStates();

                await initializeSmartChangeFilter();

                if ((node.inputMode === 'single' && node.subscriptionPattern) || (node.inputMode === 'multiple' && node.stateList.length > 0)) {
                    node.isSubscribed = true;
                    updateStatusWithValue();
                }
                node.isInitialized = true;

            } catch (error) {
                const errorMsg = error.message || 'Unknown error';

                if (errorMsg.includes('auth_failed') || errorMsg.includes('Authentication failed')) {
                    setStatus("red", "ring", "Auth failed");
                } else if (errorMsg.includes('not possible in state')) {
                    setStatus("red", "ring", "Connection failed");
                } else {
                    setStatus("yellow", "ring", "Retrying...");
                }

                node.isSubscribed = false;
            }
        }

        // Handle Input Message
        node.on('input', async function(msg) {
            // If msg.topic is missing, just trigger current values
            if (msg.topic === undefined || msg.topic === null) {
                await triggerCurrentValues();
                return;
            }
            
            let newMode = 'single';
            let newPattern = '';
            let newStateList = [];
            
            // Check for Array (Multiple States)
            if (Array.isArray(msg.topic)) {
                newMode = 'multiple';
                newStateList = msg.topic
                    .map(s => resolveEnvVar(s))
                    .map(s => s.trim())
                    .filter(s => s.length > 0);
                    
                if (newStateList.length === 0) {
                     // Empty array provided? Maybe unsubscribe everything?
                     // Or just treat as "no change" if we want to be safe?
                     // User said "Wird kein msg.topic mitgesended oder msg.topic unverändert ist..."
                     // Empty array is explicit "nothing".
                     // Let's assume empty array means "unsubscribe all".
                     // But wait, if it's empty, we can't subscribe.
                     // Let's just return for now to avoid errors.
                     return; 
                }
                
            } else if (typeof msg.topic === 'string') {
                newMode = 'single';
                newPattern = resolveEnvVar(msg.topic).trim();
                if (!newPattern) {
                     // Empty string?
                     await triggerCurrentValues();
                     return;
                }
            } else {
                // Other types? Treat as no topic change -> trigger current
                await triggerCurrentValues();
                return;
            }
            
            // Check if changed
            let hasChanged = false;
            if (newMode !== node.inputMode) {
                hasChanged = true;
            } else if (newMode === 'single') {
                if (newPattern !== node.subscriptionPattern) hasChanged = true;
            } else {
                // Compare arrays
                if (newStateList.length !== node.stateList.length) {
                    hasChanged = true;
                } else {
                    const sortedNew = [...newStateList].sort();
                    const sortedOld = [...node.stateList].sort();
                    if (JSON.stringify(sortedNew) !== JSON.stringify(sortedOld)) {
                        hasChanged = true;
                    }
                }
            }
            
            if (hasChanged) {
                node.debug(`Switching subscription to new ${newMode} configuration`);
                
                // 1. Unsubscribe old
                if (node.isSubscribed) {
                    try {
                        if (node.inputMode === 'multiple') {
                             await connectionManager.unsubscribeMultiple(
                                settings.nodeId,
                                settings.serverId,
                                Array.from(node.subscribedStates)
                            );
                        } else if (node.subscriptionPattern) {
                            await connectionManager.unsubscribe(
                                settings.nodeId,
                                settings.serverId,
                                node.subscriptionPattern
                            );
                        }
                        node.isSubscribed = false;
                    } catch (e) {
                        node.warn(`Error unsubscribing: ${e.message}`);
                    }
                }

                // 2. Update state
                node.inputMode = newMode;
                if (newMode === 'single') {
                    node.subscriptionPattern = newPattern;
                    node.stateList = [];
                    isWildcardPattern = newPattern.includes('*');
                    isSingleState = !isWildcardPattern;
                    isMultipleStates = false;
                    settings.useWildcard = isWildcardPattern;
                    settings.sendInitialValue = config.sendInitialValue && !isWildcardPattern;
                } else {
                    node.subscriptionPattern = '';
                    node.stateList = newStateList;
                    isWildcardPattern = false;
                    isSingleState = false;
                    isMultipleStates = true;
                    settings.useWildcard = false;
                    settings.sendInitialValue = config.sendInitialValue; // Allowed for multiple
                }
                
                // Reset values
                node.currentStateValues.clear();
                node.lastValue = undefined;
                node.hasReceivedValue = false;
                node.previous.clear();
                
                // 3. Subscribe new (FORCE INITIAL VALUE)
                try {
                    await subscribeToStates(true); // Force initial value on change
                    await initializeSmartChangeFilter();
                    node.isSubscribed = true;
                    updateStatusWithValue();
                } catch (e) {
                     node.error(`Error subscribing: ${e.message}`);
                     setStatus("red", "ring", "Subscription failed");
                }
            } else {
                // Not changed, just trigger current values
                await triggerCurrentValues();
            }
        });

        node.on("close", async function (removed, done) {
            node.isInitialized = false;
            node.isSubscribed = false;

            try {
                // 1) Unsubscribe first while connection is still open
                if (node.inputMode === 'multiple') {
                    await connectionManager.unsubscribeMultiple(
                        settings.nodeId,
                        settings.serverId,
                        Array.from(node.subscribedStates)
                    );
                } else if (node.subscriptionPattern) {
                    await connectionManager.unsubscribe(
                        settings.nodeId,
                        settings.serverId,
                        node.subscriptionPattern
                    );
                }

                // 2) Now unregister from events and clear status
                await NodeHelpers.handleNodeClose(node, settings, 'subscription');

            } catch (error) {
                node.warn(`Cleanup error: ${error.message}`);
            } finally {
                done();
            }
        });

        node.on("error", NodeHelpers.createErrorHandler(node, setError));

        initialize();
    }

    RED.nodes.registerType("iob-in-inject", iobInInject);
};
